/** @file   polygon.cpp
 * @brief   Implementation of Polygon class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:44 $
 * @author  Tomi Lamminsaari
 */


#include "polygon.h" // class's header file
#include "www_assert.h"


namespace WeWantWar {



///
/// Constructors, destructor
/// ========================

/** Default constructor
 */
Polygon::Polygon() :
  m_texture( - 1),
  m_polytype( POLYTYPE_FLAT )
{
  Vtx v1;
  v1.coords.set(-1,-1);
  m_originalVertices.push_back( v1 );
  v1.coords.set(-1, 1);
  m_originalVertices.push_back( v1 );
  v1.coords.set( 1, 1);
  m_originalVertices.push_back( v1 );
  v1.coords.set( 1,-1);
  m_originalVertices.push_back( v1 );
}


/** Constructs a polygon with 3 vertices
 */
Polygon::Polygon( const Vtx& rV1, const Vtx& rV2, const Vtx& rV3 ) :
  m_texture( -1 ),
  m_polytype( POLYTYPE_FLAT )
{
  m_originalVertices.push_back( rV1 );
  m_originalVertices.push_back( rV2 );
  m_originalVertices.push_back( rV3 );
}



/** Constructs a polygon with 4 vertices
 */
Polygon::Polygon( const Vtx& rV1, const Vtx& rV2, const Vtx& rV3,
                  const Vtx& rV4 ) :
  m_texture( -1 ),
  m_polytype( POLYTYPE_FLAT )
{
  m_originalVertices.push_back( rV1 );
  m_originalVertices.push_back( rV2 );
  m_originalVertices.push_back( rV3 );
  m_originalVertices.push_back( rV4 );
}



// class destructor
Polygon::~Polygon()
{
	// insert your code here
}



/** Copy constructor
 */
Polygon::Polygon( const Polygon& rP ) :
  m_texture( rP.m_texture ),
  m_polytype( rP.m_polytype ),
  m_originalVertices( rP.m_originalVertices )
{
}



/** Assignment operator
 */
Polygon& Polygon::operator = ( const Polygon& rP )
{
  if ( this != &rP ) {
    m_texture = rP.m_texture;
    m_polytype = rP.m_polytype;
    m_originalVertices = rP.m_originalVertices;
  }
  return *this;
}




///
/// Public methods
/// ==============

/** Sets the texture
 */
void Polygon::texture( int textureIndex )
{
  m_texture = textureIndex;
}



/** Sets the drawingmode
 */
void Polygon::polytype( int ptype )
{
  m_polytype = ptype;
}



/** Sets the vertices
 */
void Polygon::setVertex( int vindex, const Vtx& rV )
{
  WWW_ASSERT( vindex >= 0 );
  WWW_ASSERT( vindex < m_originalVertices.size() );
  
  m_originalVertices.at(vindex).coords = rV.coords;
  m_originalVertices.at(vindex).tcoords = rV.tcoords;
  m_originalVertices.at(vindex).c = rV.c;
}




///
/// Getter methods
/// ==============

/** Returns the vertex
 */
void Polygon::getVertex( int vindex, Vtx* pV ) const
{
  WWW_ASSERT( vindex >= 0 );
  WWW_ASSERT( vindex < m_originalVertices.size() );
  
  pV->coords = m_originalVertices.at(vindex).coords;
  pV->tcoords = m_originalVertices.at(vindex).tcoords;
  pV->c = m_originalVertices.at(vindex).c;
}



/** Returns the rotated vertex
 */
void Polygon::getRotatedVertex_ex( int vindex, V3D_f* pV, float a ) const
{
  WWW_ASSERT( vindex >= 0 );
  WWW_ASSERT( vindex < m_originalVertices.size() );
  
  const Vtx& rV = m_originalVertices.at(vindex);

  // First we rotate this vertex
  eng2d::Vec2D tmpV = rV.coords;
  tmpV.rotate( a );
  
  // And then we copy the data to another structure.
  pV->x = tmpV.x();
  pV->y = tmpV.y();
  pV->z = 1;
  
  pV->u = rV.tcoords.x();
  pV->v = rV.tcoords.y();
  
  pV->c = rV.c.allC();
}



/** Returns the number of vertices this polygon has
 */
int Polygon::vertexCount() const
{
  return m_originalVertices.size();
}



/** Returns the texture
 */
int Polygon::texture() const
{
  return m_texture;
}



/** Returns the polytype
 */
int Polygon::polytype() const
{
  return m_polytype;
}


} // end of namespace
